import { TopNav } from "@/components/nav";
import { Button, Card, CardBody, CardHeader, Container } from "@/components/ui";
import Link from "next/link";
import { getSessionOptional } from "@/lib/session";
import { prisma } from "@/lib/prisma";
import { fromMinor } from "@/lib/money";
import { transferLefaToLefa } from "@/lib/ledger";
import { GuestPay } from "@/components/guest/GuestPay";

export default async function PayRequestTokenPage({ params }: { params: { token: string } }) {
  const session = await getSessionOptional();

  const pr = await prisma.paymentRequest.findUnique({
    where: { token: params.token },
    include: { toUser: true },
  });

  async function pay() {
    "use server";
    if (!pr) throw new Error("NOT_FOUND");
    if (pr.status !== "ACTIVE") return pr.paidTxId ?? "";

    if (!session) throw new Error("UNAUTHENTICATED");

    // Execute internal LEFA transfer
    const tx = await transferLefaToLefa({
      fromUserId: session.uid,
      toEmail: pr.toUser.email,
      amountMinor: pr.amount,
      description: pr.description ?? `QR payment${pr.reference ? ` ${pr.reference}` : ""}`,
    });

    await prisma.paymentRequest.update({
      where: { id: pr.id },
      data: {
        status: "PAID",
        paidTxId: tx.transaction.id,
      },
    });

    return tx.transaction.id;
  }

  if (!pr) {
    return (
      <>
        <TopNav />
        <Container>
          <div className="mx-auto max-w-xl">
            <Card>
              <CardHeader title="Pay request" subtitle="Not found" />
              <CardBody>
                <div className="text-sm text-zinc-700">This payment request link is invalid.</div>
              </CardBody>
            </Card>
          </div>
        </Container>
      </>
    );
  }

  return (
    <>
      <TopNav />
      <Container>
        <div className="mx-auto max-w-xl">
          <Card>
            <CardHeader title="Pay request" subtitle={pr.status === "PAID" ? "Completed" : "Confirm payment"} />
            <CardBody>
              <div className="rounded-xl border border-zinc-200 p-4">
                <div className="text-xs text-zinc-600">Pay to</div>
                <div className="text-sm font-medium">{pr.toUser.email}</div>

                <div className="mt-2 text-xs text-zinc-600">Amount</div>
                <div className="text-2xl font-semibold">P {fromMinor(pr.amount)}</div>

                {pr.reference ? <div className="mt-2 text-xs text-zinc-600">Reference: {pr.reference}</div> : null}
                {pr.description ? <div className="mt-1 text-xs text-zinc-600">{pr.description}</div> : null}
                <div className="mt-3 text-xs text-zinc-500">Status: {pr.status}</div>
              </div>

              {pr.status === "PAID" ? (
                <div className="mt-4 text-sm text-green-700">Payment completed.</div>
              ) : (
                <>
                  {session ? (
                    <form action={pay} className="mt-4">
                      <Button type="submit" className="w-full">Pay with LEFA</Button>
                    </form>
                  ) : (
                    <div className="mt-4 grid gap-2">
                      <Link href={`/signin?callbackUrl=/pay/r/${pr.token}`}>
                        <Button className="w-full">Sign in to pay with LEFA</Button>
                      </Link>
                      <GuestPay mode="PAY_REQUEST" token={pr.token} />
                    </div>
                  )}
                </>
              )}
            </CardBody>
          </Card>
        </div>
      </Container>
    </>
  );
}
